/*
 * Copyright (c) 2024 SHiDAX Corporation
 */

(() => {
  /** リンク集のkintoneアプリID。 */
  const LINKS_APP_ID = 386;

  /** 代替表現のkintoneアプリID。 */
  const ALTERNATIVES_APP_ID = 384;

  /** 検索クエリを入れていないときに表示されるリンクの数。 */
  const NUM_DEFAULT_LINKS = 5;

  /** 表示されるリンクの最大数。 */
  const NUM_MAX_LINKS = 20;

  /** 検索結果に表示するかどうかを決める一致率。0から1の範囲で指定する。 */
  const MIN_SCORE = 0.5;

  /** ページを開いたとき、キャッシュが以下の期間よりも古ければ最新情報を取りに行く。 */
  const LONG_TTL = 3 * 24 * 60 * 60 * 1000;  // 3日

  /** 検索ボックスにフォーカスを当てたとき、キャッシュが以下の期間よりも古ければ最新情報を取りに行く。 */
  const SHORT_TTL = 5 * 60 * 1000;  // 5分

  /** HTML要素のID。 */
  const ELEMENT_ID = 'shidax-garoon-faq';


  /**
   * kintoneから取得してきたリンクの情報を表すデータ構造。
   *
   * @typedef {Object} LinkData
   * @property {string}   id         - レコードのID。
   * @property {string}   text       - リンクに表示する文字列。
   * @property {string}   url        - リンク先のURL。
   * @property {string[]} searchKeys - 検索用の表現リスト。textの値をnormalizeTextに通したものも含む。
   */

  /**
   * 正式な表現と代替表現（同義語などの言い換え）を表すデータ構造。
   *
   * @typedef {Object} Alternative
   * @property {string} to   - 正式名称として指定された表現。
   * @property {string} from - 考えられるほかの表現。
   */

  /**
   * kintoneから取得できる情報を表すデータ構造。
   *
   * @typedef {Object} Dataset
   * @property {LinkData[]}    links        - リンクの一覧。
   * @property {Alternative[]} alternatives - 代替表現の一覧。
   */

  /**
   * リストの中の1つのリンクを表すクラス。
   * LinkListNodeの内部で使われる。
   */
  class LinkItemNode {
    /**
     * @param {LinkData} data
     */
    constructor({ id, text, url }) {
      /** @type {string} */
      this.id = id;

      /** @type {string} */
      this.text = text;

      /** @type {string} */
      this.url = url;

      /** @type {HTMLLIElement} */
      this.element = document.createElement('li');

      const a = document.createElement('a');
      a.href = url;
      a.textContent = text;
      this.element.appendChild(a);
    }
  }

  /**
   * リンクのリストを表すクラス。
   * DOM内の必要な箇所だけを更新する機能を持っている。
   */
  class LinkListNode {
    /**
     * @param {HTMLDivElement} parentNode - 親要素のHTMLElement。
     */
    constructor(parentNode) {
      /** @type {LinkItemNode[]} */
      this._items = [];

      /** @type {HTMLUListElement} */
      this.element = parentNode.querySelector('ul');
    }

    /**
     * リストの中身を更新する。
     *
     * @param {LinkData[]} newval
     * @returns {void}
     */
    setItems(newval) {
      for (let i=0; i<Math.min(this._items.length, newval.length); i++) {
        const old = this._items[i];
        if (old.id !== newval[i].id) {
          this._items[i] = new LinkItemNode(newval[i]);
          this.element.replaceChild(this._items[i].element, old.element);
        }
      }
      for (let i=this._items.length; i<newval.length; i++) {
        this._items.push(new LinkItemNode(newval[i]));
        this.element.appendChild(this._items[i].element);
      }
      while (this._items.length > newval.length) {
        this.element.removeChild(this._items.pop().element);
      }
    }
  }

  /**
   * ガルーンFAQのUIを描画するためのクラス。
   * このクラスはUIのみで、検索ロジックは実装していない。
   */
  class RealtimeSearchNode {
    /**
     * @param {HTMLDivElement} parentNode - 親要素のHTMLElement。
     */
    constructor(parentNode) {
      /**
       * 検索クエリが更新された時のコールバック関数。
       * @type {(query: string) => void}
       */
      this.onSearch = () => {};

      /**
       * 検索窓にフォーカスが当たった時のコールバック関数。
       * @type {() => void}
       */
      this.onFocus = () => {};

      /**
       * 描画に使うHTMLElement。
       * parentNodeとしてコンストラクタに渡されたものと同じ。
       * @type {HTMLDivElement}
       */
      this.element = parentNode;

      this.input = parentNode.querySelector('input');
      this.input.addEventListener('input', (ev) => this.onSearch(this.input.value));
      this.input.addEventListener('focus', () => this.onFocus());

      this.list = new LinkListNode(parentNode);
    }

    /**
     * 準備完了フラグをセットする。
     *
     * @param {boolean} isReady - trueなら準備完了、falseならローディング中かエラー発生中。
     */
    setReady(isReady) {
      this.input.disabled = !isReady;
      this.element.classList.toggle('garoon-faq-is-ready', isReady);
    }

    /**
     * エラー等のメッセージを表示する。
     *
     * @param {string} html - 表示するメッセージの内容。
     */
    setMessage(html) {
      this.element.querySelector('.garoon-faq-message').innerHTML = html;
    }
  }

  /**
   * 代替表現の一覧を使って、想定される揺らぎのパターン一覧を作る。
   *
   * 例えば代替表現に `{ "from": "PC", "to": "パソコン" }` が含まれている場合、
   * 「PCが壊れた」を渡すと、`["PCが壊れた", "パソコンが壊れた"]` が返る。
   *
   * このときに「PCR検査をしたい」を「パソコンR検査をしたい」と変換してしまっても問題が出ないように、
   * 元の表現をそのまま使うパターンも返却している。
   *
   * @param {Alternative[]} alternatives - 代替表現の一覧。
   * @param {string}        query        - 検索クエリ。
   * @returns {string[]} 想定される揺らぎのパターン一覧。
   */
  const makeVariants = (alternatives, query) => {
    const f = (inputs) => {
      const variants = new Set();
      for (const q of inputs) {
        variants.add(q);
        for (const x of alternatives) {
          if (q.includes(x.from)) {
            variants.add(q.replaceAll(x.from, x.to));
          }
        }
      }
      return [...variants];
    };

    let variants = f([query]);
    for (let i=0; i<100; i++) {
      const v2 = f(variants);
      if (v2.length === variants.length) {
        break;
      }
      variants = v2;
    }
    return variants;
  };

  /**
   * 文字列から2-gramのセットを作成する。
   *
   * スコア計算にそのまま使えるように、重複を排除したSetとして返す。
   *
   * 空白文字を含む部分は戻り値に含まれない。
   * これにより、クエリ内の区切り文字として空白文字を含めた場合にスコアが下がってしまうことを防げる。
   *
   * @param {string} text - 2-gramを作成する元の文字列。
   * @returns {Set<string>} 作成した2-gramのセット。
   */
  const makeBigram = (text) => {
    const bigrams = new Set();
    for (let i=0; i<text.length-1; i++) {
      const bigram = text.slice(i, i+2);
      if (!/\s/.test(bigram)) {
        bigrams.add(bigram);
      }
    }
    return bigrams;
  }

  /**
   * クエリがテキストにどの程度マッチしているのかのスコアを算出する。
   * スコアが高い方がマッチしているとみなされる。1なら完璧に一致、0なら全く違う。
   *
   * 計算の仕組み：
   *  クエリと対象テキストを2文字ずつの"2-gram"に分割して、一致する割合をスコアとして使用する。
   *  クエリに含まれる2-gramが全て対象テキストに含まれる場合に1、全く含まれない場合に0になる。
   *
   *  検索クエリが1文字の場合は2-gramが使えないので、その文字が対象文字列に含まれるなら1になる。
   *
   * この関数はクエリの正規化を実施しないので、呼び出し元でnormalizeTextを実行してから渡すこと。
   *
   * @param {string} text  - 一致度を調べたい検索対象のテキスト。
   * @param {string} query - 検索クエリ。
   * @returns {number} 0から1で示した一致度。
   */
  const calcScore = (text, query) => {
    if (query.length === 1) {
      return Number(text.includes(query));
    }

    const ts = makeBigram(text);
    const qs = makeBigram(query);

    if (qs.size === 0) {
      return 0;
    }

    const count = [...qs].filter((q) => ts.has(q)).length;
    return count / qs.size;
  }

  /**
   * 与えられた半角全角や大文字小文字を統一して、検索用に正規化する。
   * Unicode正規化をしてしまうので、印字用表現としては不適切な場合があるので注意。
   *
   * @param {string} text - 正規化する文字列。
   * @returns {string} 正規化された文字列。
   */
  const normalizeText = (text) => text.normalize('NFKC').trim().toLowerCase();

  /***
   * データセットから一致度が高いリンクを検索する。
   *
   * クエリが空文字列の場合は、データセットの先頭（=最新）からNUM_DEFAULT_LINKS件を返す。
   * それ以外の場合は、クエリに一致するものを一致度の高い順にNUM_MAX_LINKS件まで返す。
   *
   * @param {Dataset} data     - 検索対象のデータセット。
   * @param {string}  query    - 検索クエリ。
   * @param {number}  minScore - 最低限のスコア。これより低いものは結果に含まない。
   * @returns {LinkData[]} 検索結果。一致度が高いものから順に並んでいる。
   */
  const search = ({ links, alternatives }, query, minScore) => {
    query = normalizeText(query);

    if (!query) {
      return links.slice(0, NUM_DEFAULT_LINKS);
    }

    const variants = makeVariants(alternatives, query);

    return links.map((link) => ({
      ...link,
      _score: Math.max(0, ...link.searchKeys.flatMap((k) => variants.map((v) => calcScore(k, v)))),
    })).filter(({ _score }) => _score >= minScore).sort((a, b) => b._score - a._score).slice(0, NUM_MAX_LINKS);
  };

  /**
   * kintoneからデータを取得する。
   * カーソルを使うので、500件以上でも取得できる。
   *
   * リクエストの詳細は以下のドキュメントを参照。
   * https://cybozu.dev/ja/kintone/docs/rest-api/records/create-cursor/
   *
   * @param {Object} req - カーソル作成APIへのリクエスト。
   * @returns {Object[]} 取得したデータのリスト。
   */
  const fetchKintone = async (req) => {
    const requestToken = await garoon.connect.kintone.getRequestToken();

    const resp = await fetch(`/k/v1/records/cursor.json?__REQUEST_TOKEN__=${requestToken}`, {
      method: 'POST',
      headers: {
        'X-Requested-With': 'XMLHttpRequest',
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(req),
    });
    if (!resp.ok) {
      throw new Error(`Failed to create cursor: ${resp.status}: ${await resp.text()}`);
    };
    const { id, totalCount } = await resp.json();
    if (totalCount === '0') {
      return [];
    }
    if (!id) {
      throw new Error(`Cursor ID not found: ${resp.status}: ${await resp.text()}`);
    }

    const result = [];

    try {
      while (true) {
        const { records, next } = await (await fetch(`/k/v1/records/cursor.json?id=${id}&__REQUEST_TOKEN__=${requestToken}`, {
          headers: { 'X-Requested-With': 'XMLHttpRequest' },
        })).json();

        result.push(...records);

        if (!next) break;
      }
    } catch(err) {
      await fetch(`/k/v1/records/cursor.json?id=${id}&__REQUEST_TOKEN__=${requestToken}`, {
        method: 'DELETE',
        headers: { 'X-Requested-With': 'XMLHttpRequest' },
      }).catch((err) => {
        console.warn('Failed to delete cursor:', err);
      });

      throw err;
    }

    return result;
  }

  /**
   * kintoneからキャッシュを使わずにデータセットを取得する。
   *
   * @returns {Dataset} 取得したデータセット。
   */
  const fetchDataset = async () => {
    const [links, alternatives] = await Promise.all([
      fetchKintone({
        app: LINKS_APP_ID,
        fields: ['$id', 'text', 'url', 'text_alternatives'],
        query: 'order by 更新日時 desc',
      }),
      fetchKintone({
        app: ALTERNATIVES_APP_ID,
        fields: ['from', 'to'],
      }),
    ]);

    return {
      links: links.map(({ $id, text, url, text_alternatives }) => ({
        id: $id.value,
        text: text.value,
        url: url.value,
        searchKeys: [
          normalizeText(text.value),
          ...text_alternatives.value.split('\n').map(normalizeText).filter((x) => x),
        ],
      })),
      alternatives: alternatives.flatMap(({ from, to }) => from.value.split('\n').map(normalizeText).filter((x) => x).map((from) => ({ from, to: normalizeText(to.value) }))),
    };
  };

  /**
   * kintoneからデータセットを取得する。
   * キャッシュがあればそれを使う。
   *
   * キャッシュが古い場合は、一度キャッシュを返してから非同期でデータセットを取得して、更新でき次第
   * 最新の情報でコールバックを呼び直すようになっている。
   * このため、コールバックは複数回呼ばれる場合がある。
   *
   * キャッシュの有効期限内であっても、この関数の戻り値として返される関数を呼べば再取得できる。
   * この機能は、検索ボックスにフォーカスを当てた時に最新情報を取りに行くために使われている。
   *
   * @param {(data: Dataset, err: Error) => void} cb - データセットを取得した後に呼び出す関数。
   * @returns {() => void} 強制的にデータセットを再取得させる関数。
   */
  const onDatasetRefresh = (cb) => {
    const cacheKey = `shidax-garoon-faq-${LINKS_APP_ID}-${ALTERNATIVES_APP_ID}`;
    const cache = (() => {
      try {
        return JSON.parse(localStorage.getItem(cacheKey) || '{}') || {};
      } catch (err) {
        console.warn('Failed to load cache:', err);
        return {};
      }
    })();

    if (cache.dataset) {
      cb(cache.dataset, null);
    }

    const refresh = () => {
      if (cache.timestamp == null || cache.timestamp + SHORT_TTL < new Date().getTime()) {
        fetchDataset().then((dataset) => {
          cache.dataset = dataset;
          cache.timestamp = new Date().getTime();
          try {
            localStorage.setItem(cacheKey, JSON.stringify(cache));
          } catch (err) {
            console.warn('Failed to save cache:', err);
          }
          cb(dataset, null);
        }).catch((err) => {
          cb(cache.dataset, err);
        });
      }
    }

    if (cache.timestamp == null || cache.timestamp + LONG_TTL < new Date().getTime()) {
      refresh();
    }

    return refresh;
  };


  const GaroonFAQ = new RealtimeSearchNode(document.getElementById(ELEMENT_ID));

  const refresh = onDatasetRefresh((dataset, err) => {
    if (err) {
      GaroonFAQ.setReady(false);
      GaroonFAQ.setMessage('データの取得に失敗しました。<br />しばらくしてから再度お試しください。');
      console.error('Failed to fetch dataset:', err);
    } else {
      GaroonFAQ.setReady(true);

      GaroonFAQ.onSearch = (query) => {
        const result = search(dataset, query, MIN_SCORE);
        GaroonFAQ.list.setItems(result);
        GaroonFAQ.setMessage(result.length > 0 ? '' : '回答が見つかりませんでした。<br />他のキーワードをお試しください。');
      };
      GaroonFAQ.onSearch(GaroonFAQ.input.value);
    }
  });
  GaroonFAQ.onFocus = () => refresh();
})();
